﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Linq;
using System.Windows.Forms;
using System.Collections.Specialized;
using System.Globalization;
using RevisionAnalyser.Global;

namespace RevisionAnalyser.Controls
{
    [Serializable]
    public partial class RevisionSlider : Control
    {
        // Static class variables
        private static short MIN_WIDTH = 400;
        private static byte HEIGHT = 60;
        private static byte PADDING = 10;
        private static byte TIMELINE_HEIGHT = 15;
        private static short DEFAULT_ZOOM = 360;
        private static short MINIMUM_ZOOM = 60;
        private static byte FONT_SIZE = 7;
        private static Font FONT = new Font("Tahoma", FONT_SIZE);
        private static Brush BRUSH_WHITE = new SolidBrush(Color.White);
        private static Brush BRUSH_LIGHT_GRAY = new SolidBrush(Color.FromArgb(230, 230, 230));
        private static Brush BRUSH_GRAY = new SolidBrush(Color.Gray);
        private static Brush BRUSH_BLACK = new SolidBrush(Color.Black);
        private static Brush BRUSH_BLUE = new SolidBrush(Color.Blue);
        private static Brush BRUSH_RED = new SolidBrush(Color.Red);
        private static Brush BRUSH_LIGHT_RED = new SolidBrush(Color.FromArgb(255, 153, 153));
        private static Brush BRUSH_GREEN = new SolidBrush(Color.LightGreen);
        private static string NO_SET_ATTACHED = "Please attach a RevisionSet to this slider";
        private static string NO_REV_ANALYSED = "No revisions analysed yet";

        // Class variables
        private int _mouseX = -1;
        private int _width = 0;
        private int _height = 0;
        private int _centerX = 0;
        private int _centerY = 0;
        private int _zoomSpeed = 60;
        private int _revCount = 0;
        private long _firstRev = 0;
        private long _lastRev = 0;
        private DateTime _startSlide;
        private DateTime _endSlide;
        private DateTime _minSlide;
        private DateTime _maxSlide;
        private double _totalDays;
        private double _totalSpace;
        private double _singleDaySpace;
        private long _selectedRev = -1;
        //private bool _sliding = false;
        //private bool _holdingCtrl = false;
        //private DateTime _startSelect;
        //private DateTime _endSelect;
        //private Lookup<int, long> _xLookup;

        private RevisionSet _revisionSet;
        public RevisionSet RevisionSet
        {
            get
            {
                return _revisionSet;
            }
            set
            {
                _revisionSet = value;
                _revisionSet.Updated += new EventHandler(_revisionSet_Updated);
                Refresh();
            }
        }

        public RevisionSlider()
        {
            InitializeComponent();
            MaximumSize = new Size(0, HEIGHT);
            MinimumSize = new Size(MIN_WIDTH, HEIGHT);
            DoubleBuffered = true;
            ResizeRedraw = true;
            MouseClick += new MouseEventHandler(RevisionSlider_MouseClick);
            //MouseDown += new MouseEventHandler(RevisionSlider_MouseDown);
            //MouseUp += new MouseEventHandler(RevisionSlider_MouseUp);
            MouseMove += new MouseEventHandler(RevisionSlider_MouseMove);
            MouseLeave += new EventHandler(RevisionSlider_MouseLeave);
            MouseWheel += new MouseEventHandler(RevisionSlider_MouseWheel);
            KeyDown += new KeyEventHandler(RevisionSlider_KeyDown);
            KeyUp += new KeyEventHandler(RevisionSlider_KeyUp);
            Resize += new EventHandler(RevisionSlider_Resize);
            RevisionSlider_Resize(null, null);
        }

        private void PaintSlider(Graphics canvas)
        {
            // Draw the background
            DrawBackground(canvas);
            // Stop here when no RevisionSet is configured
            if (RevisionSet == null)
            {
                DrawMessage(canvas, NO_SET_ATTACHED);
                return;
            }
            // Stop here when no revisions are analysed
            if (_revCount == 0) 
            {
                DrawMessage(canvas, NO_REV_ANALYSED);
                return;
            }
            // Start drawing
            //DrawSlider(canvas);
            DrawRevisions(canvas);
            DrawYearTimeline(canvas);
            DrawMouseLine(canvas);
            //CreateLookupList();
        }

        private void DrawBackground(Graphics canvas)
        {
            canvas.FillRectangle(BRUSH_LIGHT_GRAY, new Rectangle(0, 0, _width, _height));
            canvas.FillRectangle(BRUSH_BLACK, new Rectangle(0, 0, _width, 1));
        }

        /*private void DrawSlider(Graphics canvas)
        {
            double startX = GetPositionFromDate(_startSelect);
            double endX = GetPositionFromDate(_endSelect);
            if (startX < endX)
            {
                canvas.FillRectangle(BRUSH_GREEN, new Rectangle((int)startX, 0, (int)(endX - startX), HEIGHT));
            }
            else
            {
                canvas.FillRectangle(BRUSH_GREEN, new Rectangle((int)endX, 0, (int)(startX - endX), HEIGHT));
            }
        }*/

        private void DrawMouseLine(Graphics canvas)
        {
            canvas.FillRectangle(BRUSH_RED, new Rectangle(_mouseX, 1, 1, HEIGHT));
        }

        private void DrawYearTimeline(Graphics canvas)
        {
            canvas.FillRectangle(BRUSH_BLACK, new Rectangle(0, HEIGHT - PADDING - TIMELINE_HEIGHT, _width, TIMELINE_HEIGHT));
            byte innerPadding = 2;
            for (DateTime y = _startSlide.AddYears(-1); y < _endSlide; y = new DateTime(y.AddYears(1).Year, 1, 1))
            {
                if (_singleDaySpace > 0.15)
                {
                    for (int m = 1; m <= 12; m++)
                    {
                        float mPos = (float)GetPositionFromDate(new DateTime(y.Year, m, 1));
                        if (mPos > PADDING)
                        {
                            canvas.FillRectangle(BRUSH_GRAY, mPos, 1, 1, HEIGHT - PADDING - TIMELINE_HEIGHT - 1);
                            canvas.FillRectangle(BRUSH_GRAY, mPos, HEIGHT - PADDING, 1, PADDING);
                            if (_singleDaySpace > 1.0 && m > 1)
                            {
                                string txt = CultureInfo.CurrentCulture.DateTimeFormat.GetAbbreviatedMonthName(m);
                                canvas.DrawString(txt, FONT, BRUSH_GRAY, mPos, HEIGHT - PADDING - TIMELINE_HEIGHT + innerPadding);
                            }
                        }
                    }
                }
                float yPos = (float)GetPositionFromDate(y);
                if (yPos > PADDING)
                {
                    canvas.FillRectangle(BRUSH_BLACK, yPos, 1, 1, HEIGHT - PADDING - TIMELINE_HEIGHT - 1);
                    canvas.FillRectangle(BRUSH_BLACK, yPos, HEIGHT - PADDING, 1, PADDING);
                    canvas.DrawString(y.Year.ToString(), FONT, BRUSH_WHITE, yPos, HEIGHT - PADDING - TIMELINE_HEIGHT + innerPadding);
                }
            }
        }

        private void DrawRevisions(Graphics canvas)
        {
            foreach (Revision revision in RevisionSet.GetRevisionList())
            {
                if (revision.Time >= _startSlide && revision.Time <= _endSlide)
                {
                    DrawRevision(canvas, revision.ID, revision.Time);
                }
            }
        }

        private int DrawRevision(Graphics canvas, long rev, DateTime revDate)
        {
            float xPos = (float)GetPositionFromDate(revDate);
            if (rev == _selectedRev)
            {
                canvas.FillRectangle(BRUSH_LIGHT_RED, xPos - 3, 1, 3, _height);
            }
            Brush b = (rev == _selectedRev) ? BRUSH_RED : BRUSH_BLUE;
            canvas.FillRectangle(b, xPos - 3, (HEIGHT - PADDING - TIMELINE_HEIGHT) / 2, 3, 3);
            return (int)xPos;
        }

        private void DrawMessage(Graphics canvas, string message)
        {
            SizeF measure = canvas.MeasureString(message, FONT);
            canvas.DrawString(message, FONT, BRUSH_BLACK, _centerX - (measure.Width / 2), _centerY - (measure.Height / 2));
        }

        private DateTime GetDateFromPosition(double pos)
        {
            return _startSlide.AddDays((pos - PADDING) / _singleDaySpace);
        }

        private DateTime GetDateFromPosition(double pos, bool safe)
        {
            DateTime date = GetDateFromPosition(pos);
            if (safe && date >= _endSlide)
            {
                return _endSlide;
            }
            return date;
        }

        private double GetPositionFromDate(DateTime date)
        {
            double days = date.Subtract(_startSlide).TotalDays;
            if (days > 0)
            {
                return PADDING + (days * _singleDaySpace);
            }
            return PADDING;
        }

        private void RecalculateZoomSpeed()
        {
            _zoomSpeed = (int)_endSlide.Subtract(_startSlide).TotalDays / 10;
        }

        /*private void CreateLookupList()
        {
            _xLookup = (Lookup<int, long>)_revisionSet.GetRevisionList().ToLookup(p => p.XPosition, p => p.ID);
        }*/

        protected override void OnPaint(PaintEventArgs pe)
        {
            base.OnPaint(pe);
            PaintSlider(pe.Graphics);
        }

        void RevisionSlider_Resize(object sender, EventArgs e)
        {
            _width = ClientSize.Width;
            _height = ClientSize.Height;
            _centerX = _width / 2;
            _centerY = _height / 2;
            _totalSpace = _width - (2 * PADDING);
            if (_revCount > 0)
            {
                _singleDaySpace = _totalSpace / _totalDays;
                RecalculateZoomSpeed();
            }
        }

        void _revisionSet_Updated(object sender, EventArgs e)
        {
            _revCount = RevisionSet.RevisionCount;
            _totalDays = 0;
            _selectedRev = -1;
            if (_revCount > 0)
            {
                _firstRev = RevisionSet.FirstRevision;
                _lastRev = RevisionSet.LastRevision;
                _startSlide = RevisionSet.GetRevision(_firstRev).Time;
                _endSlide = RevisionSet.GetRevision(_lastRev).Time;
                while (_totalDays <= DEFAULT_ZOOM)
                {
                    _startSlide = _minSlide = _startSlide.AddDays(-60);
                    _endSlide = _maxSlide = _endSlide.AddDays(60);
                    _totalDays = Convert.ToInt32(_endSlide.Subtract(_startSlide).TotalDays) + 1;
                }
                RecalculateZoomSpeed();
                _singleDaySpace = _totalSpace / _totalDays;
            }
            else
            {
                _firstRev = 0;
                _lastRev = 0;
                _startSlide = DateTime.Now;
                _endSlide = DateTime.Now;
                //_startSelect = DateTime.Now;
                //_endSelect = DateTime.Now;
                _singleDaySpace = 0;
            }
            Invalidate();
        }

        void RevisionSlider_MouseClick(object sender, MouseEventArgs e)
        {
            if (_revCount == 0)
            {
                return;
            }
            long closestRev = -1;
            double closestDiff = _width;
            foreach (Revision revision in RevisionSet.GetRevisionList())
            {
                if (revision.Time >= _startSlide && revision.Time <= _endSlide)
                {
                    double xPos = GetPositionFromDate(revision.Time);
                    double diff = Math.Abs(xPos - e.X);
                    if (diff < closestDiff)
                    {
                        closestRev = revision.ID;
                        closestDiff = diff;
                    }
                }
            }
            if (_selectedRev != closestRev)
            {
                RevisionSelected(sender, new RevisionEventArgs(closestRev));
            }
            _selectedRev = closestRev;
            Invalidate();
        }

        void RevisionSlider_MouseDown(object sender, MouseEventArgs e)
        {
            if (_revCount == 0)
            {
                return;
            }
            //_startSelect = GetDateFromPosition(e.X, true);
            //_sliding = true;
        }

        void RevisionSlider_MouseUp(object sender, MouseEventArgs e)
        {
            if (_revCount == 0)
            {
                return;
            }
            /*_endSelect = GetDateFromPosition(e.X, true);
            if (_endSelect.CompareTo(_startSelect) < 0)
            {
                DateTime tmp = _startSelect;
                _startSelect = _endSelect;
                _endSelect = tmp;
            }
            if (_endSelect.Subtract(_startSelect).TotalHours < 24)
            {
                _endSelect = _startSelect.AddDays(1);
            }*/
            //_sliding = false;
        }

        void RevisionSlider_MouseMove(object sender, MouseEventArgs e)
        {
            if (_revCount == 0)
            {
                return;
            }
            Focus();
            _mouseX = e.X;
            /*if (_sliding)
            {
                _endSelect = GetDateFromPosition(e.X, true);
            }*/            
            Invalidate();
        }

        void RevisionSlider_MouseWheel(object sender, MouseEventArgs e)
        {
            if (_revCount == 0)
            {
                return;
            }
            /*if (_holdingCtrl && _startSelect != null && _endSelect != null)
            {
                if (e.Delta > 0)
                {
                    if (_startSelect > _startSlide)
                    {
                        _startSelect = _startSelect.AddDays(-1);
                    }
                    if (_endSelect < _endSlide)
                    {
                        _endSelect = _endSelect.AddDays(1);
                    }
                }
                else
                {
                    if (_endSelect.Subtract(_startSelect).TotalDays >= 3)
                    {
                        _startSelect = _startSelect.AddDays(1);
                        _endSelect = _endSelect.AddDays(-1);
                    }
                }
            }
            else
            {*/
                double pos = e.X / _totalSpace;
                if (e.Delta > 0)
                {
                    if (_totalDays >= MINIMUM_ZOOM + _zoomSpeed)
                    {
                        _startSlide = _startSlide.AddDays(pos * _zoomSpeed);
                        _endSlide = _endSlide.AddDays(-1 * (1 - pos) * _zoomSpeed);
                    }
                }
                else
                {
                    _startSlide = _startSlide.AddDays(-1 * pos * _zoomSpeed);
                    _endSlide = _endSlide.AddDays((1 - pos) * _zoomSpeed);
                    if (_startSlide < _minSlide)
                    {
                        _startSlide = _minSlide;
                    }
                    if (_endSlide > _maxSlide)
                    {
                        _endSlide = _maxSlide;
                    }
                }
                _totalDays = Convert.ToInt32(_endSlide.Subtract(_startSlide).TotalDays) + 1;
                _singleDaySpace = _totalSpace / _totalDays;
            //}
            Invalidate();
        }

        void RevisionSlider_MouseLeave(object sender, EventArgs e)
        {
            if (_revCount == 0)
            {
                return;
            }
            _mouseX = -1;
            Invalidate();
        }

        void RevisionSlider_KeyUp(object sender, KeyEventArgs e)
        {
            if (_revCount == 0)
            {
                return;
            }
            //Cursor = Cursors.Default;
            //_holdingCtrl = false;
        }

        void RevisionSlider_KeyDown(object sender, KeyEventArgs e)
        {
            if (_revCount == 0)
            {
                return;
            }
            /*if (e.Control)
            {
                Cursor = Cursors.SizeWE;
                _holdingCtrl = true;
            }
            else
            {
                Cursor = Cursors.Default;
                _holdingCtrl = false;
            }*/
        }

        public event RevisionEventHandler RevisionSelected;

        //public event RevisionEventHandler RevisionMouseHover;

        //public event RevisionEventHandler RevisionMouseLeave;

        public delegate void RevisionEventHandler(object sender, RevisionEventArgs e);

        public class RevisionEventArgs : EventArgs
        {
            private long _revision;
            public long Revision
            {
                get
                {
                    return _revision;
                }
                set
                {
                    _revision = value;
                }
            }

            public RevisionEventArgs(long revision)
            {
                _revision = revision;
            }
        }
    }
}
